program parfit
  !
  !     PARFIT - A FORTRAN CODE TO FIND BEST FIT PARAMETERS BETWEEN
  !     HAZMAP MODEL AND DEPOSIT DATA
  !
  !     Authors: Giovanni Macedonio, Antonio Costa
  !
  !     Note: parfit was firstly derived from hazmap version 2.3.1
  !     This version is compatible with hazmap 2.4
  !
  use config
  use parmod
  use airprop
  use windutils
  use settling
  use safeop
  implicit none
  character(len=80) :: message
  !
  integer :: npts                       ! Number of grid point
  !
  ! Command line arguments
  integer :: narg                ! Number of arguments in the command line
  integer :: lenpref
  !
  !.... File names
  character(len=80) :: prefix  ! Files prefix
  character(len=80) :: finp    ! Input file  (data)
  character(len=80) :: fout    ! Output file (ascii)
  character(len=80) :: fwnd    ! Output wind file
  character(len=80) :: fcmp    ! Output comparison file
  character(len=80) :: fchi    ! Output file for chi2 info
  character(len=80) :: ground_thickness='ground_thickness.inp' ! Default
  character(len=80) :: ground_spectra='ground_spectra.inp'     ! Default
  character(len=80) :: column_spectrum='column_spectrum.inp'   ! Default
  character(len=80) :: input_wind_file='wind.dat'              ! Default
  !
  ! Unit numbers
  integer :: ninp              ! Input file  (data)
  integer :: ngrd              ! Input ground thicknesses
  integer :: ncsp              ! Input column spectrum
  integer :: ngsp              ! Input ground spectra
  integer :: ncmp              ! Output comparison file
  integer :: nchi              ! Output chi2

  real(8) :: xvent,yvent,zvent,xgtmp,ygtmp

  character :: cflag,cflagm
  logical :: checkhull         ! Check for deposit maxima in the convex hull

  ! Flags for domain boundaries
  integer, parameter :: IFLAG0=1
  integer, parameter :: IFLAG1=2
  integer, parameter :: IFLAG2=4
  integer, parameter :: IFLAG3=8
  integer, parameter :: IFLAG4=16
  integer, parameter :: IFLAG5=32

  real(8) :: totmm          ! Initial guess mass (for initialization)
  real(8) :: rhodre         ! Density Rock Equivalent
  real(8) :: htropo         ! Height of tropopause
  integer  :: ntropo        ! Index of Tropopause layer
  integer, allocatable :: iwork1(:),iwork2(:) ! Work vector for subr. convex
  integer, allocatable :: indss(:)       ! Work vector for subroutine convex
  integer, allocatable :: in(:)          ! Work vector for subroutine convex
  integer, allocatable :: il(:)          ! Work vector for subroutine convex
  integer, allocatable :: ih(:)          ! Work vector for subroutine convex
  real(8), allocatable :: xhull(:),yhull(:)   ! Vertices of the convex hull
  real(8), allocatable :: toutm(:)
  real(8), allocatable :: wground(:)
  real(8), allocatable :: out(:,:)            ! Deposit loadings
  real(8), allocatable :: xg(:,:)             ! Ground section coordinate

  integer :: iwndm                  ! Index of best hazwind
  real(8) :: wndmin,wndmax,wndstep  ! Minimum, maximum, step for wind int.
  real(8) :: dirmin,dirmax,dirstep  ! Minimum, maximum, step for wind direction
  integer :: nwnd,ndir              ! Number of wind intensities and direction
  real(8) :: deg2rad,rad2deg        ! Conversion degrees <-> radians
  real(8) :: cdmin,cdmax,cdstep     ! Diffusion coefficient: min,max,step
  real(8), allocatable :: rmoss(:)  ! Observed deposit (temporary variable)
  real(8), allocatable :: rmmod(:)  ! Modelled deposit (temporary variable)
  real(8), allocatable :: wvel(:)   ! Wind profile
  real(8), allocatable :: dthick(:) ! Deposit thicknesses
  real(8), allocatable :: ddens(:)  ! Deposit densities
  real(8), allocatable :: ground(:) ! Deposit loadings
  real(8), allocatable :: pdiam(:)  ! Particle diameter
  real(8), allocatable :: pdens(:)  ! Particle density
  real(8), allocatable :: ppsi(:)   ! Particle shape factor

  real(8), allocatable :: fsj(:,:)  ! Fraction of particles in the sections
  character(len=6), allocatable :: labgrd(:)    ! Label of cell with deposit
  character(len=6), allocatable :: labhul(:)    ! Label vertices of convex hull
  character(len=6), allocatable :: labgsp(:)    ! Label of cell with spectra
  character(len=256) :: tstring     ! Temporary string
  real(8) :: fvcorr                 ! Function for correction of vset(z)

  integer :: ncd
  real(8) :: hcolmin,hcolmax,hcolstep
  real(8) :: suz1min,suz1max,suz1step
  real(8) :: suz2min,suz2max,suz2step
  integer :: nhcol
  integer :: modew
  integer :: rwindfile
  integer :: ifgsp
  integer :: iflch                ! Flag: select convex hull
  integer :: ifenl                ! Flag: enlarge the convex hull
  real(8) :: belthull             ! Thickness of the belt around convex hull
  integer :: iflwgt               ! Use nearest points for spectra
  integer :: ifchi
  real(8), allocatable :: fp(:)   ! Particle spectrum in the column (fraction)
  real(8), allocatable :: fpm(:)  ! Particle spectrum in the column (saved)
  integer :: nsuz1,nsuz2
  real(8) :: hlayer=-1d0          ! Distance between the horizontal layers
  integer :: i,j,k,ik,ind
  integer :: nptsp    ! Number of ground sections with spectrum
  real(8) :: dzw
  integer, allocatable :: ifclass(:)  ! Include particle type
  integer, allocatable :: ifclasm(:)  ! ifclass saved
  real(8), allocatable :: totout(:)
  real(8), allocatable :: weight(:)
  integer :: ifound
  integer :: ntotsp        ! Number of sections with granulometric spectrum
  integer :: kmin
  integer :: indmin
  real(8) :: dist,dismin   ! Temporary variables
  integer :: nhull
  real(8) :: dir
  real(8) :: hcol
  integer :: iwnd
  integer :: idir
  real(8) :: wnd
  integer :: ihcol
  real(8) :: suz1,suz2
  integer :: isuz1,isuz2
  integer :: itest, ntest
  integer :: iflag
  real(8) :: rmost
  integer :: inside   ! Integer function
  integer :: ins
  real(8) :: fmass
  real(8) :: fmout
  real(8) :: xmass,ymass
  integer :: icd
  real(8) :: beta
  real(8) :: cgrd
  integer :: iflagm
  real(8) :: wndm,dirm,betam,hcolm,cdiffm,suz1m,suz2m,rchi2m
  real(8) :: sumvs
  real(8) :: cout
  real(8) :: rchi2
  real(8) :: fmin
  integer :: ifvofz             ! Flag: settling velocity is a function of Z
  integer :: vmodel             ! Settling velocity model
  integer :: smodel=1           ! Suzuki model
  real(8) :: rhoa,pres,temp     ! Atmosphere density, pressure and temperature
  real(8) :: sigma,delta,theta  ! Air dens,pres,temp respect reference
  real(8) :: visca              ! Air viscosity
  integer :: ierr
  integer :: nctypes            ! Number of considered particles types
  integer :: nsrcmax            ! Maximum number of point sources
  real(8) :: srcstep            ! Vertical step between sources
  !
  type(hazwinds) :: winds       ! Winds in hazmap format
  !
  ! Namelists
  namelist /wind/ rwindfile,htropo,wndmin,wndmax,wndstep,dirmin,dirmax,dirstep
  namelist /turbulence/ cdmin,cdmax,cdstep
  namelist /column/ hcolmin,hcolmax,hcolstep,hlayer,suz1min,suz1max,suz1step, &
       suz2min,suz2max,suz2step
  namelist /vent/ xvent,yvent,zvent
  namelist /flags/ modew,iflch,ifenl,belthull,iflwgt,vmodel,ifvofz,ifchi, &
       ifgsp,smodel
  namelist /files/ ground_thickness,ground_spectra,column_spectrum, &
       input_wind_file
  !
  ! Get command line argument
  narg = command_argument_count()
  if(narg /= 1) then
     write(*,'(''Parfit '',a,1x,''('',a,'')'')') version, git_version
     write(*,'(''Usage: parfit parfit[.inp]'')')
     call exit(1)
  end if
  call get_command_argument(1,prefix)

  ! Generate file names
  lenpref = len_trim(prefix)
  if(lenpref > 4) then
     if(prefix(lenpref-3:lenpref) == '.inp') prefix = prefix(1:lenpref-4)
  end if
  finp = trim(prefix)//'.inp'
  fout = trim(prefix)//'.out'
  fcmp = trim(prefix)//'.cmp'
  fwnd = trim(prefix)//'.wnd'
  fchi = trim(prefix)//'_chi2.out'
  !
  !.... Set guess mass
  totmm  = 1e10               ! This is arbitrary
  rhodre = 2600.              ! Rock density (for DRE)
  !
  ! Write version number
  write(message,'(''Parfit version: '',a)') version
  call info(message)
  !
  !.... Open/Read input data file
  !
  call getfreeunit(ninp)
  if(ninp < 0) call abend('Cannot get file unit (ninp)')
  open(ninp,file=finp,status='old',err=2002)
  read(ninp,wind,end=3001,err=3001)
  rewind(ninp)
  read(ninp,turbulence,end=3002,err=3002)
  rewind(ninp)
  read(ninp,column,end=3003,err=3003)
  rewind(ninp)
  read(ninp,vent,end=3004,err=3004)
  rewind(ninp)
  read(ninp,flags,end=3005,err=3005)
  rewind(ninp)
  read(ninp,files,end=50,err=3006)  ! This block is optional
  50 continue
  close(ninp)
  !
  !.... Open output file
  !
  call getfreeunit(nout)
  if(nout < 0) call abend('Cannot get file unit (nout)')
  open(nout,file=fout,status='unknown',err=2004)

  nwnd   = nint(safediv(wndmax-wndmin,wndstep))
  dirmin = deg2rad(dirmin)       ! Convert to radiants
  dirmax = deg2rad(dirmax)       ! Convert to radiants
  dirstep= deg2rad(dirstep)      ! Convert to radiants
  ndir   = nint(safediv((dirmax-dirmin),dirstep))
  ncd    = nint(safediv((cdmax-cdmin),cdstep))
  nhcol  = nint(safediv(hcolmax-hcolmin,hcolstep))
  ! Next line was commented starting from version 2.2.1
  ! hstep  = hcolstep  ! <= OLD, keep commented: set vertical step = hcolstep
  ! Variable hstep becomes hlayer
  nsuz1  = nint(safediv(suz1max-suz1min,suz1step))
  nsuz2  = nint(safediv(suz2max-suz2min,suz2step))
  ! Check
  if(hlayer <= 0d0) then
     write(*,'(''Error: hlayer not set'')')
     call exit(1)
  end if
  !
  ! Wind and z-layers
  !
  if(rwindfile == 0) then   ! WIND is generated internally
     !.... Estimate the number of layers up to tropopause
     ntropo = nint(safediv(htropo,hlayer))
     !.... Generate zeta (wind layers) levels
     !
     dzw = htropo/ntropo          ! Thickness of wind layers
     nzlev = ceiling(hcolmax/dzw) ! Max. number of wind layers (may be < ntropo)
     ntropo = min(ntropo,nzlev)   ! Don't need wind layers above nzlev
     ! Set wind layers
     allocate(zeta(0:nzlev))
     do i = 0,nzlev
        zeta(i) = i*dzw
     enddo
     ! Allocate memory for wind profile
     allocate(wvel(0:nzlev))
     ! Generate normalized wind profile
     call genprotowind(nzlev,zeta,htropo,ntropo,wvel)
     ! windx and windy are set in the loop on parameters
     allocate(windx(0:nzlev))
     allocate(windy(0:nzlev))
     !
  else  ! WIND is read from file
     !
     ! Read wind file
     write(message,'(''Using wind file: '',a)') trim(input_wind_file)
     call info(message)
     !
     call hazwinds_read(winds, input_wind_file)
     write(message,'(''Number of read wind profiles: '',i8)') winds%nwinds
     call info(message)
     ndir = 0   ! Does not loop on directions
     nzlev = winds%nz
     nwnd = winds%nwinds
     allocate(windx(0:nzlev))
     allocate(windy(0:nzlev))
     allocate(zeta(0:nzlev))
     zeta(0:nzlev) = winds%zeta(0:winds%nz)
  end if
  !
  ! Evaluate DZ (used by parcumula)
  allocate(dz(0:nzlev))
  do i = 1,nzlev
     dz(i) = zeta(i)-zeta(i-1)
  enddo
  dz(0) = dz(1)
  !
  if(ifchi==1) then
     call getfreeunit(nchi)
     if(nchi < 0) call abend('Cannot get file unit (nchi)')
     open(nchi,file=fchi,status='unknown')
  endif
  !
  !
  if(ifgsp==0) then       ! Read velocity spectrum from column_spectrum.inp
     call getfreeunit(ncsp)
     if(ncsp < 0) call abend('Cannot get file unit (ncsp)')
     open(ncsp,file=column_spectrum,status='old',err=2006)
     !.... Read number of particle types
     read(ncsp,*) ntypes
     !
     allocate(pdiam(ntypes))
     allocate(pdens(ntypes))
     allocate(ppsi(ntypes))
     allocate(fp(ntypes))
     !
     ! Read diameter, density, shape, percentage
     do j=1,ntypes
        read(ncsp,*) pdiam(j),pdens(j),ppsi(j),fp(j)
     enddo
     ! Convert to fractions
     fp = fp/100.
     !
     close(ncsp)
     write(message,'(''Read column spectrum from file: '',a)') trim(column_spectrum)
     call info(message)
  endif
  !
  ! Estimate maximum number of source points (for memory allocation)
  nsrcmax = ceiling(hcolmax/hlayer)
  !
  !
  ! Allocate memory
  allocate(dxdist(0:nzlev))
  allocate(dydist(0:nzlev))
  allocate(dvinv(0:nzlev))
  !
  !.... Open/Read ground thickness and deposit density
  !
  call getfreeunit(ngrd)
  if(ngrd < 0) call abend('Cannot get file unit (ngrd)')
  open(ngrd,file=ground_thickness,status='old',err=2003)
  ! Count the number of points (lines)
  npts=0
  do
     read(ngrd,'(a)',end=100,err=2008) tstring
     if(tstring(1:1) == '#') cycle   ! Skip comments
     npts = npts + 1
  end do
100 continue
  rewind(ngrd)
  !
  ! Allocate memory
  allocate(labgrd(npts))
  allocate(xg(2,npts))
  allocate(dthick(npts))
  allocate(ddens(npts))
  allocate(ground(npts))
  allocate(totout(npts))
  allocate(weight(npts))
  allocate(toutm(npts))
  allocate(wground(npts))
  allocate(in(npts))
  allocate(ih(npts))
  allocate(il(npts))
  allocate(xhull(npts))
  allocate(yhull(npts))
  allocate(labhul(npts))
  allocate(zsrc(nsrcmax))
  allocate(iwork1(npts))
  allocate(iwork2(npts))
  iwork1=0       ! Clear vector
  iwork2=0       ! Clear vector
  !
  i=1
  tstring = ''   ! Clear
  do
     read(ngrd,'(a)',end=101) tstring
     if(tstring(1:1) == '#') cycle  ! Skip comments
     read(tstring,*,end=2008,err=2008) labgrd(i),xgtmp,ygtmp,dthick(i),ddens(i)
     xg(1,i) = (xgtmp-xvent)
     xg(2,i) = (ygtmp-yvent)
     ground(i) = dthick(i)*ddens(i)
     in(i)=i             ! Needed by subroutine convex
     i=i+1
  enddo
101 continue
  if(i-1 /= npts) call abend('Problems reading ground thickness file')
  !
  close(ngrd)
  !
  !.... Check whether two sections have the same number
  do i=1,npts-1
     do j=i+1,npts
        if(labgrd(i)==labgrd(j)) then
           write(message,'(''***ERROR: Two or more sections have the &
                &same number: '',a)') trim(labgrd(i))
           call abend(message)
        endif
     enddo
  enddo
  !
  write(message,'(''Number of ground sections: '',i4)') npts
  call info(message)
  !
  !.... Generate weights for chi2 (deposit)
  call depw(npts,ground,weight,modew)
  !
  !.... Open/Read ground particles spectrum
  !
  if(ifgsp==1) then
     write(message,'(''Using ground spectra from file: '',a)') trim(ground_spectra)
     call info(message)
     call getfreeunit(ngsp)
     if(ngsp < 0) call abend('Cannot get file unit (ngsp)')
     open(ngsp,file=ground_spectra,status='old',err=2005)
     !
     !.... Read number of particle types
     read(ngsp,*) ntypes
     !
     allocate(pdiam(ntypes))
     allocate(pdens(ntypes))
     allocate(ppsi(ntypes))
     allocate(fp(ntypes))
     !
     ! Read particles diameter, density and shape
     do j=1,ntypes
        read(ngsp,*) pdiam(j),pdens(j),ppsi(j)
     enddo
     !
     ! Read spectra in the ground sections
     read(ngsp,*) nptsp
     !
     if(nptsp > npts) then
        write(message,'(''Number of ground sections with spectrum is &
             &greater than NPTS='',i5)') npts
        call abend('Check input files')
     endif
     !
     allocate(labgsp(nptsp))        ! Allocate only for nptsp points
     allocate(fsj(npts,ntypes))     ! Allocate for all npts points
     allocate(indss(npts))          ! Allocate for all npts points
     indss = 0                      ! Clear indss
     !
     ! Read percentage of each particle
     do i=1,nptsp
        read(ngsp,*) labgsp(i)
        do j=1,ntypes
           read(ngsp,*) fsj(i,j)
        enddo
     enddo
     !
     ! Convert to fractions
     write(message,'(''Number of sections with spectrum: '',i3)') nptsp
     call info(message)
     do i=1,nptsp
        write(message,'(''Section: '',a,'' Sum: '',f9.5,'' (should be 100)'')')&
             labgsp(i),sum(fsj(i,:))
        call info(message)
     enddo
     fsj = fsj/100.  ! Convert to mass fraction
     !
     close(ngsp)
     !
     !
     !  CONSISTENCY TEST
     !
     !.... Check whether two spectra have the same section number
     do i=1,nptsp-1
        do j=i+1,nptsp
           if(labgsp(i)==labgsp(j)) then
              write(message,'(''Two or more spectra have the same section &
                   &number:'',a)') trim(labgsp(i))
              call abend(message)
           endif
        enddo
     enddo
     !.... Check consistency between spectra and sections
     do i=1,nptsp  ! Loop on the ground spectra
        ifound=0
        do j=1,npts ! Loop on the ground points
           if(labgsp(i)==labgrd(j)) then
              indss(i)=j
              ifound=1
              exit
           endif
        enddo
        if(ifound==0) then
           write(message,'(''Cannot find section corresponding to spectrum='',a)') trim(labgsp(i))
           call abend(message)
        endif
     enddo
  endif
  !
  !.... To compute weights of ground sections, guess spectrum for ground
  !     sections not listed in file ground_spectra.inp
  !     Here: assume that the spectrum is equal to that of the nearest
  !     point listed in ground_spectra.inp
  !
  if(ifgsp==1.and.iflwgt==1) then
     ntotsp=nptsp
     do i=1,npts
        kmin=1
        indmin = indss(kmin)
        dismin=(xg(1,i)-xg(1,indmin))**2+(xg(2,i)-xg(2,indmin))**2
        do k=1,nptsp
           ind = indss(k)
           if(ind==i) goto 25 ! Already have spectrum
           dist = (xg(1,i)-xg(1,ind))**2+(xg(2,i)-xg(2,ind))**2
           if(dist < dismin) then
              dismin = dist
              kmin=k
              indmin=ind
           endif
        enddo
        ntotsp=ntotsp+1     ! Guessed spectra are located in fsj(i,j)
        indss(ntotsp)=i     ! starting from i=nptsp+1
        do j=1,ntypes
           fsj(ntotsp,j) = fsj(kmin,j)
        enddo
25      continue
     enddo                  ! After this loop ntotsp is equal to npts
  endif
  !
  ! Allocate memory
  allocate(srcmas(nsrcmax))
  allocate(b(nsrcmax,ntypes))
  allocate(rex(nsrcmax,ntypes))
  allocate(rey(nsrcmax,ntypes))
  allocate(rmoss(ntypes))
  allocate(rmmod(ntypes))
  allocate(fpm(ntypes))
  allocate(ifclass(ntypes))
  allocate(ifclasm(ntypes))
  allocate(vvj(0:nzlev,ntypes))
  allocate(out(npts,ntypes))
  !
  !.... Computes settling velocity in the layers
  !
  write(message,'(''Settling velocity model: '',a)') trim(vset_model_name(vmodel))
  call info(message)
  !
  do i=0,nzlev
     !
     if(ifvofz==0) then  ! Settling velocity evaluated at sea level
        rhoa = rhoa0
        pres = pres0
        temp = temp0
     elseif(ifvofz==1) then  ! Vsettl. is a function of Z (Standard atmosphere)
        call atmosphere(zeta(i),sigma,delta,theta)
        rhoa = sigma*rhoa0
        pres = delta*pres0
        temp = theta*temp0
     elseif(ifvofz==2) then  ! Vsettl. is a function of Z (use fvcorr)
        rhoa = rhoa0
        pres = pres0
        temp = temp0
     else
        call abend('Invalid flag IFVOFZ in the input file')
     endif
     call sutherland(temp,visca)
     do j=1,ntypes
        call vsettl(pdiam(j),pdens(j),rhoa,visca,vvj(i,j),vmodel,ppsi(j),ierr)
        if(ierr==1) then
           call abend('Invalid settling velocity model')
        elseif(ierr == 2) then
           write(message,'(''***Error in subroutine vsettl'')')
           call info(message)
           write(message,'(''Convergence not reached'')')
           call info(message)
           write(message,'(''Diameter  ='',1x,e12.5)') pdiam(j)
           call info(message)
           write(message,'(''Density   ='',1x,e12.5)') pdens(j)
           call info(message)
           write(message,'(''Shape fact='',1x,e12.5)') ppsi(j)
           call info(message)
           write(message,'(''Air dens  ='',1x,e12.5)') rhoa
           call info(message)
           write(message,'(''Air visc  ='',1x,e12.5)') visca
           call info(message)
           write(message,'(''Vset model='',1x,i2)') vmodel
           call abend('STOP')
        endif
     enddo
  enddo
  !
  ! Recompute vsettling(Z) if use fvcorr
  if(ifvofz==2) then
     do i=1,nzlev
        do j=1,ntypes
           vvj(i,j) = vvj(0,j)*fvcorr(vvj(0,j),zeta(i))
        enddo
     enddo
  endif
  !
  if(smodel==1) then
     call info('Suzuki model: integrated')
  else if(smodel==2) then
     call info('Suzuki model: classical')
  else
     call abend('Invalid Suzuki model (smodel)')
  end if
  !
  !.... Initialize wground
  wground = 1.0
  !
  !     Generate convex hull of the ground section points
  !
  call convex(npts,xg,npts,in,iwork1,iwork2,ih,nhull,il)
  ! Copy the convex hull in vector (xhull,yhull)
  ik=il(1)
  do i=1,nhull
     j=ih(ik)
     xhull(i) = xg(1,j)
     yhull(i) = xg(2,j)
     labhul(i) = labgrd(j)
     ik=il(ik)
  enddo
  !
  ! Enlarge convex hull (if request)
  if(ifenl==1) call enlarge(nhull,xhull,yhull,belthull)
  !
  !.... Select method for handling settling velocity classes
  if(iflch==1) then
     checkhull=.true.       ! Check if ground points are inside hull
     if(ifgsp==1) then
        write(message,'(''Check for maxima inside convex hull: YES'')')
        call info(message)
        write(message,'(''List of vertices of the convex hull'')')
        call info(message)
        if(ifenl==1) then
           do i=1,nhull
              write(message,'(a6,1x,(2(1x,f9.1)),2x,''(enlarged convex hull)'')') &
                   trim(labhul(i)),xvent+xhull(i),yvent+yhull(i)
              call info(message)
           enddo
        else
           do i=1,nhull
              write(message,'(a,1x,(2(1x,f9.1)))') trim(labhul(i)),   &
                   xvent+xhull(i),yvent+yhull(i)
              call info(message)
           enddo
        endif
     endif
  else
     do i=1,ntypes
        ifclass(i) = 1      ! Include particle type
     enddo
     checkhull=.false.
     if(ifgsp==1) then
        write(message,'(''Check for maxima inside convex hull: NO'')')
        call info(message)
     endif
  endif
  !
  if(ifgsp==0) then         ! Read spectrum from parfit.inp
     do i=1,ntypes
        ifclass(i) = 1      ! Set particle type flags (default)
     enddo
     if(checkhull) then
        write(message,'(''WARNING: Check hull N/A when spectrum is static'')')
        call info(message)
     endif
  endif
  !
  if(rwindfile == 0) then
     ntest = (nsuz1+1)*(nsuz2+1)*(nhcol+1)*(nwnd+1)*(ndir+1)*(ncd+1)
  else
     ntest = (nsuz1+1)*(nsuz2+1)*(nhcol+1)*nwnd*(ncd+1)
  end if
  write(message,'(''Number of tests: '',i8)') ntest
  call info(message)
  !
  !     LOOPS FOR BEST CHI2
  !
  iflag=0
  rchi2m = HUGE(1d0)       ! Initialize minimum chi2 to a very big number
  !
  !.... LOOP on Suzuki coefficients
  itest = 0   ! Test number
  do isuz1=0,nsuz1
     if(isuz1==0.or.isuz1==nsuz1) then
        call setbits(iflag,IFLAG0) ! 1st loop -> Bit 0 (2**0=1)
     else
        call clearbits(iflag,IFLAG0)
     endif
     suz1 = suz1min+isuz1*suz1step
     do isuz2=0,nsuz2
        if(isuz2==0.or.isuz2==nsuz2) then
           call setbits(iflag,IFLAG1) ! 2nd loop -> Bit 1 (2**1=2)
        else
           call clearbits(iflag,IFLAG1)
        endif
        suz2 = suz2min+isuz2*suz2step
        !
        !.... LOOP on COLUMN HEIGHT
        do ihcol=0,nhcol
           if(ihcol==0.or.ihcol==nhcol) then
              call setbits(iflag,IFLAG2) ! 3rd loop -> Bit 2 (2**2=4)
           else
              call clearbits(iflag,IFLAG2)
           endif
           hcol = hcolmin+ihcol*hcolstep
           nsrc = nint((hcol-zvent)/hlayer)
           srcstep = (hcol-zvent)/nsrc
           do j=1,nsrc
              zsrc(j) = zvent+j*srcstep ! Used by subroutine parcumula
           enddo
           !
           if(smodel==1) then
              ! Integrated suzuki
              call isuzuki (srcmas,nsrc,suz1,suz2)
           else if(smodel==2) then
              ! Classical suzuchi (not-integrated)
              call suzuki (srcmas,nsrc,suz1,suz2)
           else
              call abend('Invalid Suzuki model (smodel)')
           end if
           !
           !.... LOOP on WIND (Intensity and direction)
           do iwnd=0,nwnd   ! *** Loop on wind speed/wind profiles
              if(rwindfile == 0) then
                 if(iwnd==0.or.iwnd==nwnd) then
                    call setbits(iflag,IFLAG3) ! 4th loop -> Bit 3 (2**3=8)
                 else
                    call clearbits(iflag,IFLAG3)
                 end if
                 wnd = wndmin+iwnd*wndstep ! Wind at tropopause
              else
                 ! Get wind profile iwnd from the winds list
                 if(iwnd == 0) cycle
                 call hazwinds_get(winds, iwnd, windx, windy, wdate)
              end if
              !
              do idir=0,ndir ! Loop on wind direction (ndir=0 if rwindfile=1)
                 if(rwindfile == 0) then
                    if(idir==0.or.idir==ndir) then
                       call setbits(iflag,IFLAG4) ! 5th loop -> Bit 4 (2**4=16)
                    else
                       call clearbits(iflag,IFLAG4)
                    endif
                    dir = dirmin+idir*dirstep
                    !.... Generate wind profile (from prototype)
                    call genwind(nzlev,wnd,dir,wvel,windx,windy)
                 end if
                 !
                 !.... Find center of mass of gaussian deposits
                 !
                 call parcumula
                 !
                 !.... Estimate settling velocity spectrum
                 !
                 if(ifgsp==1) then ! Use file spectra.inp
                    fmin  = 0.
                    fmout = 0.
                    nctypes = 0
                    do j=1,ntypes
                       xmass = 0.
                       ymass = 0.
                       do i=1,nsrc
                          fmass = srcmas(i)
                          xmass = xmass + fmass*rex(i,j)
                          ymass = ymass + fmass*rey(i,j)
                       enddo
                       ! Check if deposit maxima are inside the perimeter (convex hull) of
                       !     the ground points
                       if(checkhull) then
                          ins=inside(nhull,xhull,yhull,xmass,ymass)
                          if(ins==1) then
                             ifclass(j) = 0 ! Exclude this particle type
                             fmout = fmout + fp(j)
                          else
                             ifclass(j) = 1 ! Include this particle type
                             fmin = fmin + fp(j)
                             nctypes = nctypes + 1
                          endif
                       else
                          nctypes = nctypes + 1
                       endif
                    enddo
                 else  ! Spectra from parfit.inp
                    nctypes = ntypes
                 endif
                 !
                 !.... LOOP of DIFFUSION COEFFICIENT
                 do icd=0,ncd
                    if(icd==0.or.icd==ncd) then
                       call setbits(iflag,IFLAG5) ! 6th loop -> Bit 5 (2**5=32)
                    else
                       call clearbits(iflag,IFLAG5)
                    endif
                    cdiff = cdmin+icd*cdstep
                    !
                    !     ******* Begin of the inner loop ********
                    !
                    !
                    ! Compute deposit in the given sections for p-type
                    !
                    call depclas(npts,xg,out)
                    !
                    ! Estimate mass in the column for each particle type
                    if(ifgsp==1) then    ! Use file ground_spectra.inp
                       do j=1,ntypes
                          rmoss(j) = 0.
                          rmmod(j) = 0.
                          do i=1,nptsp
                             if(ifclass(j)==1) then
                                ind = indss(i)
                                rmoss(j)=rmoss(j) + ground(ind)*fsj(i,j)
                                rmmod(j)=rmmod(j) + totmm*out(ind,j)
                             endif
                          enddo
                       enddo
                       !.... Fit spectrum
                       rmost = 0.
                       do j=1,ntypes
                          if(ifclass(j)==1) then
                             fp(j) = safediv(rmoss(j),rmmod(j))
                             rmost = rmost + fp(j)
                          endif
                       enddo
                       !.... Normalize
                       do j=1,ntypes
                          if(ifclass(j)==1) then
                             fp(j) = safediv(fp(j),rmost)
                          endif
                       enddo
                    endif
                    !
                    !.... Compute total mass
                    do i=1,npts
                       totout(i) = 0.
                       do j=1,ntypes
                          if(ifclass(j)==1) then
                             totout(i) = totout(i) + out(i,j)*fp(j)
                          endif
                       enddo
                    enddo
                    !
                    ! Compute weigths of ground sections
                    ! (wgrounds were initialized=1)
                    !     if(checkhull.and.ifgsp==1.and.iflwgt==1)then
                    if(ifgsp==1.and.iflwgt==1)then
                       do i=1,ntotsp
                          ind = indss(i)
                          wground(ind)=0.
                          do j=1,ntypes
                             if(ifclass(j)==1) then
                                wground(ind)=wground(ind)+fsj(i,j)
                             endif
                          enddo
                       enddo
                    endif
                    !
                    !.... Normalize deposit (best fit with total mass)
                    cout=0.
                    cgrd=0.
                    do i=1,npts
                       cout = cout+totout(i)
                       cgrd = cgrd+ground(i)*wground(i)
                    enddo
                    !
                    beta = safediv(cgrd,cout)
                    !
                    do i=1,npts
                       totout(i) = beta*totout(i)
                    enddo
                    !
                    !.... Compute chi2
                    call chi2(rchi2,npts,totout,ground,weight)
                    !
                    if(ifchi==1) then
                       write(nchi,80) suz1,suz2,hcol,wnd,rad2deg(dir), &
                            cdiff,beta,rchi2,nctypes
80                     format(8(1x,e12.5),1x,i3)
                    endif
                    !
                    if(iflag==0) then
                       cflag='I'
                    else
                       cflag='B'
                    endif
                    !
                    !.... Check for minimum chi2
                    if(rchi2 < rchi2m) then
                       iflagm = iflag
                       suz1m  = suz1
                       suz2m  = suz2
                       betam = beta
                       hcolm = hcol
                       cdiffm = cdiff
                       rchi2m = rchi2
                       cflagm = cflag
                       if(rwindfile == 0) then
                          wndm  = wnd
                          dirm  = dir
                       else
                          iwndm = iwnd
                       end if
                       do j=1,ntypes
                          ifclasm(j) = ifclass(j)
                          if(ifclass(j)==1) fpm(j) = fp(j)
                       enddo
                       do i=1,npts
                          toutm(i) = totout(i)
                       enddo
                    endif
                    ! itest = itest + 1   ! Increment the test number
                    !
                    !     **************  End of inner loop ***************
                    !
                 enddo
              enddo
           enddo
        enddo
     enddo
  enddo
  !
  !
  !.... WRITE OUTPUT FILE
  !
  write(message,'(''FLAG:      '',a)') cflagm
  call info(message)
  write(message,'(''CHI2('',i1,''):  '',g12.5)') modew,rchi2m
  call info(message)
  write(message,'(''TOTMASS:  '',e12.5)') betam
  call info(message)
  write(message,'(''VOL(DRE): '',g12.5,'' (km3)'')') betam*1E-9/rhodre
  call info(message)
  if(rwindfile==0) then
     write(message,'(''VEL-WIND: '',f7.3)') wndm
     call info(message)
     write(message,'(''DIR-WIND: '',f7.3)') rad2deg(dirm)
     call info(message)
  end if
  write(message,'(''HCOL:     '',f7.1)') hcolm
  call info(message)
  write(message,'(''CSUZ1:    '',f6.3)') suz1m
  call info(message)
  write(message,'(''CSUZ2:    '',f6.3)') suz2m
  call info(message)
  write(message,'(''CDIFF:    '',f7.1)') cdiffm
  call info(message)
  write(message,'('' Diameter     dens. sphe.   wt%     # P-TYPE  Vset (at sea level)'')')
  call info(message)
  !
  sumvs=0.0
  do j=1,ntypes
     if(ifclasm(j)==1) then
        if(100.*fpm(j) >= 1e-4) then
           sumvs=sumvs+100.*fpm(j)          ! Convert fractions to wt%
           write(message,110) pdiam(j),pdens(j),ppsi(j),100.*fpm(j),j,vvj(0,j)
           call info(message)
        end if
     endif
  enddo
  write(message,111) sumvs
  call info(message)
110 format(e12.5,1x,f6.1,1x,f5.3,1x,f8.4,2x,'#',2x,i4,1x,g12.5)
111 format('SUM:',1x,f7.3)
  !
  !
  if(cflagm=='B') then
     write(message,'(''WARNING: Min. chi2 lies on parameters borders'')')
     call info(message)
     if(and(iflagm,IFLAG0)==IFLAG0) then
        write(message,*) 'BDRY: CSUZ1',suz1m
        call info(message)
     end if
     if(and(iflagm,IFLAG1)==IFLAG1) then
        write(message,*) 'BDRY: CSUZ2',suz2m
        call info(message)
     end if
     if(and(iflagm,IFLAG2)==IFLAG2) then
        write(message,*) 'BDRY: COLUMN HEIGHT',hcolm
        call info(message)
     end if
     if(rwindfile==0) then
        if(and(iflagm,IFLAG3)==IFLAG3) then
           write(message,*) 'BDRY: WIND SPEED    ',wndm
           call info(message)
        end if
        if(and(iflagm,IFLAG4)==IFLAG4) then
           write(message,*) 'BDRY: WIND DIRECTION',rad2deg(dirm)
           call info(message)
        end if
     end if
     if(and(iflagm,IFLAG5)==IFLAG5) then
        write(message,*) 'BDRY: DIFF. COEFF.',cdiffm
        call info(message)
     end if
  endif
  !
  !.... Write comparison file
  !
  call getfreeunit(ncmp)
  if(ncmp < 0) call abend('Cannot get file unit (ncmp)')
  open(ncmp, file=fcmp, status='unknown', err=2007)
  write(ncmp,'(''# Label     X        Y        obs_load     sim_load     dist_vent'')')
  do i=1,npts
     write(ncmp,120) labgrd(i),xvent+xg(1,i),yvent+xg(2,i),ground(i),&
          toutm(i),sqrt(xg(1,i)**2+xg(2,i)**2)
  enddo
120 format(a6,1x,f10.2,1x,f10.2,1x,e12.5,1x,e12.5,1x,f10.2)
  close(ncmp)
  write(nterm,*) 'Wrote cmp    file (ASCII): '//trim(fcmp)
  !
  !.... Write wind file
  !
  if(rwindfile==0) then
     call genwind(nzlev,wndm,dirm,wvel,windx,windy)
     call wriwind(fwnd,nzlev,zeta,windx,windy,wdate)
  else
     call hazwinds_write(winds, fwnd, iwndm)
  end if
  !
  write(nterm,*) 'Wrote wind   file (ASCII): '//trim(fwnd)
  !
  !     Compute positions of gaussian centers of mass
  !
  call parcumula
  !
  nsrc = nint((hcolm-zvent)/hlayer)
  call suzuki (srcmas,nsrc,suz1m,suz2m)
  write(message,'(5x,''PT Vset(sealev)'',7x,''X_mass'',6x,''Y_mass'',4x,''Distance'')')
  call info(message)
  do j=1,ntypes
     xmass = 0.
     ymass = 0.
     do i=1,nsrc
        fmass = srcmas(i)
        xmass = xmass + fmass*rex(i,j)
        ymass = ymass + fmass*rey(i,j)
     enddo
     ! Distance from the vent
     dist = sqrt(xmass**2+ymass**2)
     !
     ! Georeference the center of mass
     xmass = xmass + xvent
     ymass = ymass + yvent
     !
     if(checkhull) then
        if(ifclasm(j)==1) then
           write(message,'(''IN '',i3,3x,g12.5,3(1x,f12.1))') &
                j,vvj(0,j),xmass,ymass,dist
           call info(message)
        else
           write(message,'(''OUT'',i3,3x,g12.5,3(1x,f12.1))') &
                j,vvj(0,j),xmass,ymass,dist
           call info(message)
        endif
     else
        write(message,'(3x,i3,3x,g12.5,3(1x,f12.1))') &
             j,vvj(0,j),xmass,ymass,dist
        call info(message)
     endif
  enddo

  !
  !
  close(nout)
  write(nterm,*) 'Wrote output file (ASCII): '//trim(fout)
  if(ifchi==1) then
     close(nchi)
     write(nterm,*) 'Wrote output file (ASCII): '//trim(fchi)
  endif
  !
  call exit(0)
  !
  !     Errors
  !
2002 write(message,*) 'Cannot open file: '//trim(finp)
  call abend(message)
2003 write(message,*) 'Cannot open file: '//trim(ground_thickness)
  call abend(message)
2004 write(message,*) 'Cannot open file: '//trim(fout)
  call abend(message)
2005 write(message,*) 'Cannot open file: '//trim(ground_spectra)
  call abend(message)
2006 write(message,*) 'Cannot open file: '//trim(column_spectrum)
  call abend(message)
2007 write(message,*) 'Cannot open file: '//trim(fcmp)
  call abend(message)
2008 write(message,*) 'Reading file: '//trim(ground_thickness)
  call abend(message)
3001 write(message,*) 'Cannot read block WIND in file: '//trim(finp)
  call abend(message)
3002 write(message,*) 'Cannot read block TURBULENCE in file: '//trim(finp)
  call abend(message)
3003 write(message,*) 'Cannot read block COLUMN in file: '//trim(finp)
  call abend(message)
3004 write(message,*) 'Cannot read block VENT in file: '//trim(finp)
  call abend(message)
3005 write(message,*) 'Cannot read block FLAGS in file: '//trim(finp)
  call abend(message)
3006 write(message,*) 'Cannot read block FILES in file: '//trim(finp)
  call abend(message)
end program parfit
